﻿#pragma once

#ifdef __cplusplus
#define EXTERN_C extern "C"
#else
#define EXTERN_C
#endif

#ifndef RVCHANDEYE_HEAD
#ifdef _WIN32
#if defined(RVCHANDEYE_EXPORT)
#define RVCHANDEYE_HEAD EXTERN_C __declspec(dllexport)
#else
#define RVCHANDEYE_HEAD EXTERN_C __declspec(dllimport)
#endif
#else
#define RVCHANDEYE_HEAD EXTERN_C
#endif
#endif

#ifdef _WIN32
#define CALLING_CONVENTION __stdcall
#else
#define CALLING_CONVENTION
#endif

// All APIs are valid until September 1, 2025. If the API is used beyond this date, a failure error will be returned.

RVCHANDEYE_HEAD void CALLING_CONVENTION DetectConcentricCircles2D(const char* imageFilePath,
                                                                  int* resultNum,
                                                                  float resultPixelXy[200]);

// If there are 2D circle center coordinates, but the corresponding point cloud circle center has hole, 
// then the 3D circle center coordinates are NaN
RVCHANDEYE_HEAD void CALLING_CONVENTION DetectConcentricCircles3D(const char* imageFilePath,
                                                                  const char* pointCloudFilePath,
                                                                  int* resultNum,
                                                                  float resultPixelXy[200],
                                                                  float resultPointXyz[300]);

RVCHANDEYE_HEAD void CALLING_CONVENTION DetectCaliboard2D(const char* imageFilePath,
                                                          const float camera_intrinsic[9],
                                                          const float camera_distortion[5],
                                                          const int caliboard_pattern_size_width,
                                                          const int caliboard_pattern_size_height,
                                                          float resultPixelXy[200]);

RVCHANDEYE_HEAD void CALLING_CONVENTION DetectCaliboard3D(
    const char* imageFilePath, const char* pointCloudFilePath, const float camera_intrinsic[9],
    const float camera_distortion[5], const int caliboard_pattern_size_width,
    const int caliboard_pattern_size_height,
    const float caliboard_circle_center_standard_3d_distance_step, float resultPixelXy[200],
    float resultPointXyz[300], float& measuring_distance, float& error_percentage);
/**
 * @brief algorithm parameter
 * type: robot pose type: 0: euler angle, 1: wxyz, 2: xyzw, 3: matrix
 * markerType: 0: Asymmetric black background and white circle calibration board, support 7*11, 4*11, 4*5
 *             1: concentric circle.
 *             If you use HandEyeCalibrationTcpTouch(), ignore this parameter.
 * isPointCloudMm: whether the unit of point cloud is mm
 * isPoseMm: whether the unit of robot pose is mm
 * isPoseDegreeL: whether the robot pose is in degree
 * isEyeInHand: whether the camera is in robot gripper
 * dataMask: data mask, 0: not use, 1: use.
 */
struct HandEyeParam {
    int poseType;
    int markerType;
    bool isPointCloudMm;
    bool isPoseMm;
    bool isPoseDegree;
    bool isEyeInHand;
    bool dataMask[100];
};

/**
 * @brief If eye in hand, the result is camera to gripper coordinate system. If eye to hand, the
 *        result is camera to base coordinate system.
 * translationResult: x y z
 * eulerAngleResult: If poseType equals euler angle, this field has result.
 * quaternionResult: w x y z or x y z w. If poseType equals quaternion, this field has result.
 * matrix: 4x4 transformation matrix
 * success2D: 0: failed, 1: success, 2: not exist image file. If you use HandEyeCalibrationTcpTouch(), ignore this field.
 * success3D: 0: failed, 1: success, 2: not exist point cloud file. If you use HandEyeCalibrationTcpTouch(), ignore this field.
 * error: each data error.
 * totalMeanError: total mean error of all data.
 */
struct HandEyeResult {
    double translationResult[3];
    double eulerAngleResult[3];
    double quaternionResult[4];
    double matrix[16];
    int success2D[100];
    int success3D[100];
    double error[100];
    double totalMeanError;
};

/**
 * @brief Perform hand-eye calibration using concentric circles or calibration board.
 * @return
 * 0: success
 * -1: failed. Because input parameters are invalid, such as folder is nullptr, robotPoseFileName is
 * nullptr, params is invalid.
 * -2: failed. Because the data in the folder is not invalid, such as the lack of point cloud files
 * or lack of image files.
 * -3: failed. Because the number of point cloud files is not equal to the number of image files.
 * -4: failed. Because the data in the robotPoseFileName is invalid.
 * -5: failed. Because the data number in the robotPoseFileName is not equal to the number of point cloud
 * files.
 * -6: failed. Because fail to detect 2D or 3D concentric circles or calibration board in some files. See success2D
 * and success3D in HandEyeResult.
 */
RVCHANDEYE_HEAD int CALLING_CONVENTION HandEyeCalibrationMarker(const char* folder,
                                                                const char* robotPoseFileName,
                                                                const HandEyeParam& params,
                                                                HandEyeResult& r);

/*
 * @brief Perform hand-eye calibration using tcp touch data. If eye to hand, robotPoseFilePath can be nullptr.
 * @return
 * 0: success
 * -1: failed. Because input parameters are invalid, such as cameraXyzFilePath is nullptr, params is invalid.
 * -2: failed. Because the data in cameraXyzFilePath is invalid.
 * -3: failed. Because the data in tcpXyzFilePath is invalid.
 * -4: failed. Because the data number in cameraXyzFilePath is not equal to the data number in tcpXyzFilePath.
 * -5: failed. Because the data in robotPoseFilePath is invalid.
 * -6: failed. Because the data number in cameraXyzFilePath is not equal to the data number in robotPoseFilePath.
 * -7: failed. Because the data in robotPoseFilePath needs to include 2 or more robot poses.
 */
RVCHANDEYE_HEAD int CALLING_CONVENTION HandEyeCalibrationTcpTouch(const char* cameraXyzFilePath,
                                                                  const char* robotPoseFilePath,
                                                                  const char* tcpXyzFilePath,
                                                                  const HandEyeParam& params,
                                                                  HandEyeResult& result);